#ifndef CE2_AES_H
#define CE2_AES_H

/*
 * All the includes that are needed for code using this module to
 * compile correctly should be #included here.
 */
#include "CE2_public.h"

#ifdef __cplusplus
extern "C"
{
#endif

  /*
   *  Object % CE2_AES.h    : %
   *  State           :  %state%
   *  Creation date   :  Wed Nov 17 16:42:30 2004
   *  Last modified   :  %modify_time%
   */
  /** @file
   *  \brief This file contains all of the enums and definitions that are used for the 
   *         CE2 AES APIs, as well as the APIs themselves. 
   *
   *  \version CE2_AES.h#1:incl:13
   *  \author adams
   *  \remarks Copyright (C) 2005 by Discretix Technologies Ltd.
   *           All Rights reserved
   */

/************************ Defines ******************************/

/** 
@brief - a definition describing the low level Engine type ( SW , Hardware , Etc )
*/

/* The size of the IV or counter buffer */
#define CE2_AES_IV_COUNTER_SIZE_IN_WORDS 4
#define CE2_AES_IV_COUNTER_SIZE_IN_BYTES (CE2_AES_IV_COUNTER_SIZE_IN_WORDS * sizeof(DxUint32_t))

/* The maximum size of the AES KEY in words and bytes */
#define CE2_AES_KEY_MAX_SIZE_IN_WORDS 8
#define CE2_AES_KEY_MAX_SIZE_IN_BYTES (CE2_AES_KEY_MAX_SIZE_IN_WORDS * sizeof(DxUint32_t))

/* The AES block size in bytes */
#define CE2_AES_BLOCK_SIZE_IN_BYTES 16

/* The AES block size in words */
#define CE2_AES_BLOCK_SIZE_IN_WORDS 4

/* The AES_WRAP minimum data size in bytes  (one 64-bits block)  */
#define CE2_AES_WRAP_DATA_MIN_SIZE_IN_BYTES   8

/* The AES_WRAP maximum data size in bytes:  2^28 */
#define CE2_AES_WRAP_DATA_MAX_SIZE_IN_BYTES   0x10000000

/* The CE2_AES_WRAP block size in bytes and in words */
#define CE2_AES_WRAP_BLOCK_SIZE_IN_BYTES 8
#define CE2_AES_WRAP_BLOCK_SIZE_IN_WORDS  (CE2_AES_WRAP_BLOCK_SIZE_IN_BYTES / sizeof(DxUint32_t))

/* The CE2_AES_CCM special definitions */
#define CE2_AESCCM_ENGINE_TYPE  LLF_AESCCM_ENGINE_TYPE

#define CE2_AESCCM_KEY_SIZE_WORDS           4
#define CE2_AESCCM_NONCE_MIN_SIZE_BYTES     7
#define CE2_AESCCM_NONCE_MAX_SIZE_BYTES    13
#define CE2_AESCCM_MAC_MIN_SIZE_BYTES       4
#define CE2_AESCCM_MAC_MAX_SIZE_BYTES      16
#define CE2_AESCCM_QFIELD_MIN_SIZE_BYTES    2
#define CE2_AESCCM_QFIELD_MAX_SIZE_BYTES    8
#define CE2_AESCCM_NQ_BLOCK_SIZE_BYTES     (CE2_AESCCM_NONCE_MAX_SIZE_BYTES+CE2_AESCCM_QFIELD_MIN_SIZE_BYTES)

/* AES CCM encrypt-decrypt mode */
#define CE2_AESCCM_Decrypt   CE2_AES_Decrypt
#define CE2_AESCCM_Encrypt   CE2_AES_Encrypt

/************************ Enums ********************************/

/* Enum defining the user's key size argument */
typedef enum
{
   CE2_AES_Key128BitSize   = 0,
   CE2_AES_Key192BitSize   = 1,
   CE2_AES_Key256BitSize   = 2,
   CE2_AES_Key512BitSize   = 3, 

   CE2_AES_KeySizeNumOfOptions
   
}CE2_AES_KeySize_t;  

/* Enum defining the Encrypt or Decrypt operation mode */
typedef enum 
{
   CE2_AES_Encrypt = 0,
   CE2_AES_Decrypt = 1,
   
   CE2_AES_EncryptNumOfOptions

}CE2_AES_EncryptMode_t;

/* Enum defining the AES operation mode */
typedef enum
{
   CE2_AES_ECB_mode          = 0,
   CE2_AES_CBC_mode          = 1,
   CE2_AES_MAC_mode          = 2,
   CE2_AES_CTR_mode          = 3,
   CE2_AES_XCBC_MAC_mode     = 4,
   CE2_AES_CMAC_mode         = 5, 
   
   CE2_AES_NumOfModes

}CE2_AES_OperationMode_t;

/************************ Typedefs  ****************************/

/* Defines the IV counter buffer */
typedef DxUint8_t CE2_AES_IvCounter_t[CE2_AES_IV_COUNTER_SIZE_IN_BYTES];

/* Define AES tweak Buffer */
typedef CE2_AES_IvCounter_t CE2_AES_XTS_Tweak_t;

/* Defines the AES key buffer */
typedef DxUint8_t CE2_AES_Key_t[CE2_AES_KEY_MAX_SIZE_IN_BYTES];

/* Defines the AES MAC result maximum size buffer */
typedef DxUint8_t CE2_AES_MAX_MAC_RESULT_t[CE2_AES_IV_COUNTER_SIZE_IN_BYTES];

/* Defines the AES_CCM key buffer */
typedef DxUint32_t CE2_AESCCM_Key_t[CE2_AESCCM_KEY_SIZE_WORDS];

/************************ Public Variables **********************/


/************************ Public Functions **********************/                    

/****************************************************************************************************/
/**
****************************************************************
* Function Name: 
*  DX_AES
*
* @param IVCounter_ptr [in] - This parameter is the buffer of the IV or 
*                  counters on mode CTR.
*                  In ECB, XCBC, CMAC mode this parameter is not used.
*                  In CBC and MAC modes this parameter should contain the IV values.
*                  In CTR mode this value should contain the init counter.
*                  In XCBC and CMAC modes it may be NULL
*  @param Key_ptr [in] - A pointer to the user's key buffer.
*  @param KeySize [in] - The size of the KEY used by the AES: 128, 192 or 256 bits, 
*            as defined in the enum.
*  @param EncryptDecryptFlag [in] - A flag specifying whether the AES should perform an 
*                       Encrypt operation (0) or a Decrypt operation (1). 
*                       In XCBC and CMAC modes it must be 0.
*  @param OperationMode [in] - The operation mode: ECB, CBC, MAC, CTR, XCBC (PRF and 96), CMAC.
*  @param DataIn_ptr [in] - The pointer to the buffer of the input data to the AES. 
*               The pointer's value does not need to be word-aligned.
*  @param DataInSize [in] - The size of the input data (must be not 0 and must be multiple 
*               of 16 bytes, besides XCBC and CMAC).
*  @param DataOut_ptr [out] - The pointer to the buffer of the output data from the AES. 
*                The pointer does not need to be aligned to 32 bits. 
*
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*   - CE2_AES_DATA_IN_PTR_ERROR
*   - CE2_AES_DATA_OUT_PTR_ERROR
*   - CE2_AES_DATA_IN_SIZE_ERROR
*   - CE2_AES_IV_COUNTER_PTR_ERROR
*   - CE2_AES_KEY_PTR_ERROR
*   - CE2_AES_KEY_SIZE_ERROR
*   - CE2_AES_OPERATION_MODE_ERROR
*   - CE2_AES_ENCRYPT_DECRYPT_FLAG_ERROR
*
* \brief \b 
* Description:
*  This function is used to operate the AES machine in one 
*  integrated operation.
*
*  \b 
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_AES 
*     for AES with using LibTomCrypt.
***************************************************************/

CE2CIMPORT_C CE2Error_t  DX_AES(CE2_AES_IvCounter_t     IVCounter_ptr,
                              CE2_AES_Key_t           Key_ptr,         
                              CE2_AES_KeySize_t       KeySize,
                              CE2_AES_EncryptMode_t   EncryptDecryptFlag,
                              CE2_AES_OperationMode_t OperationMode,
                              DxUint8_t                *DataIn_ptr,
                              DxUint32_t               DataInSize,
                              DxUint8_t                *DataOut_ptr );     





/**************************************************************************
 *	              CE2_AES_Wrap function                                  *
 **************************************************************************/
/**
****************************************************************
* Function Name: 
*  CE2_AES_Wrap
*
* @param DataIn_ptr [in] - A pointer to plain text data to be wrapped
*               NOTE: Overlapping between the data input and data output buffer
*               is not allowed, except the inplace case that is legal . 			
* @param DataInLen [in]  - Length of data in bytes. DataLen must be multiple of 
*               8 bytes and  must be in range [16,  2^28].
* @param KeyData [in] - A pointer to  key data (key encryption key - KEK). 
* @param KeySize [in] - Enumerator variable, defines length of key.
* @param WrapDataOut_ptr [out] - A pointer to buffer for output of wrapped data.
* @param WrapDataLen_ptr [in/out] - A pointer to a buffer for input of size of 
*                    user passed buffer and for output actual 
*                    size of unwrapped data in bytes. Buffer size must 
*                    be not less than DataLen+CE2_AES_WRAP_BLOCK_SIZE_IN_BYTES.                         
*
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*   - CE2_AES_WRAP_ILLEGAL_DATA_PTR_ERROR
*   - CE2_AES_WRAP_DATA_LENGTH_ERROR
*   - CE2_AES_WRAP_KEY_PTR_ERROR
*   - CE2_AES_WRAP_KEY_LENGTH_ERROR                  
*   - CE2_AES_WRAP_ILLEGAL_WRAP_DATA_PTR_ERROR      
*   - CE2_AES_WRAP_ILLEGAL_WRAP_DATA_LEN_PTR_ERROR  
*   - CE2_AES_WRAP_ILLEGAL_WRAP_DATA_LENGTH_ERROR
*
* \brief \b 
* Description:
*   The CE2_AES_Wrap function implements the following algorithm 
*   (rfc3394, Sept. 2002)
*  Inputs:  Plaintext DataIn, n 64-bit values {P1, P2, ..., Pn}, 
*   KeyData, K (the KEK).                
*  Outputs: Ciphertext, WrapDataOut (n+1) 64-bit values {C0, C1, ..., Cn}.
*
*  \b 
* Algorithm:
* -# Verify input parameters for validity;
* -# Call low level function LLF_AES_Wrap 
*    for AES wraping with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C  CE2Error_t CE2_AES_Wrap (
                               DxUint8_t            *DataIn_ptr,      /*in*/   
                               DxUint32_t            DataInLen,       /*in*/
                               CE2_AES_Key_t        KeyData,         /*in*/
                               CE2_AES_KeySize_t    KeySize,         /*in*/
                               DxUint8_t            *WrapDataOut_ptr, /*out*/
                               DxUint32_t           *WrapDataLen_ptr  /*in/out*/ ); 



/**************************************************************************
 *	              CE2_AES_Uwnrap function                                *
 **************************************************************************/
/**
****************************************************************
* Function Name: 
*  CE2_AES_Unwrap
*
* @param WrapDataIn_ptr [in] - A pointer to wrapped data to be unwrapped 
*                   NOTE: Overlapping between the data input and 
*                   data output buffer is not allowed, except the 
*                   inplace case that is legal . 			
* @param WrapDataInLen [in] - Length of wrapped data in bytes. DataLen must be 
*                  multiple of 8 bytes and  must be in range [24, 2^29].
* @param KeyData [in] - A pointer to  key data (key encryption key - KEK). 
* @param KeySize [in] - Enumerator variable, defines length of key.
* @param DataOut_ptr [out] - A pointer to buffer for output of unwrapped data.
* @param DataOutLen_ptr [in/out] - A pointer to a buffer for input of size of user 
*                   passed buffer and for output of actual size of 
*                   unwrapped data in bytes. DataOutLen must be 
*                   multiple of 8 bytes and must be not less than 
*                   WrapDataInLen - CE2_AES_WRAP_BLOCK_SIZE_IN_BYTES.
*
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*   - CE2_AES_UNWRAP_ILLEGAL_WRAP_DATA_PTR_ERROR          
*   - CE2_AES_UNWRAP_WRAP_DATA_LENGTH_ERROR          
*   - CE2_AES_UNWRAP_KEY_PTR_ERROR
*   - CE2_AES_UNWRAP_KEY_LEN_ERROR                   
*   - CE2_AES_UNWRAP_ILLEGAL_DATA_PTR_ERROR         
*   - CE2_AES_UNWRAP_ILLEGAL_DATA_LEN_PTR_ERROR      
*   - CE2_AES_UNWRAP_ILLEGAL_DATA_LENGTH_ERROR      
*
* \brief \b 
* Description:
*   The CE2_AES_Unwrap function performs inverse AES_Wrap transformation 
*   and implements the following algorithm (rfc3394, Sept. 2002): 
*  Inputs:  Ciphertext, (n+1) 64-bit values {C0, C1, ..., Cn}, and
*   K  - key (the KEK).
*  Outputs: Plaintext, n 64-bit values {P1, P2, ..., Pn}.
*
*  \b 
* Algorithm:
* -# Verify input parameters for validity;
* -# Call low level function LLF_AES_Unwrap 
*    for AES unwraping with using LibTomCrypt.
***************************************************************/

CE2CIMPORT_C  CE2Error_t CE2_AES_Unwrap(
                                   DxUint8_t            *WrapDataIn_ptr, /*in*/   
                                   DxUint32_t            WrapDataInLen,    /*in*/
                                   CE2_AES_Key_t        KeyData,        /*in*/
                                   CE2_AES_KeySize_t    KeySize,        /*in*/ 
                                   DxUint8_t            *DataOut_ptr,    /*out*/
                                   DxUint32_t           *DataOutLen_ptr     /*in/out*/ );
/**
****************************************************************
* Function Name: 
*  DX_AES_GenerateTweakValue
*
* @param Tweak_dst [out] - This parameter is a return value from generator Tweak Value.
* @param Key_ptr [in] - This parameter is a key used for generator.
* @param KeySize [in] - Size of key.
* @param SectorNumber [in] - sector number.
*
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*   - CE2_XTS_AES_ILLEGAL_TWEAK_PTR_ERROR
*   - CE2_XTS_AES_ILLEGAL_KEY_PTR_ERROR
*   - CE2_XTS_AES_ILLEGAL_SECTOR_NUMBER_PTR_ERROR
*
* \brief \b 
* Description:
*  Used for generating sector "tweak"
*
**  \b 
* Algorithm:
*  -# Verify input parameters;
*  -# Call low level function LLF_AES_GenerateTweakValue().
***************************************************************/
CE2CIMPORT_C CE2Error_t DX_AES_GenerateTweakValue(CE2_AES_XTS_Tweak_t Tweak_dst,
                                                  CE2_AES_Key_t       Key_ptr,
												  CE2_AES_KeySize_t   KeySize,
                                                  CE2_AES_XTS_Tweak_t SectorNumber );
/**
****************************************************************
* Function Name: 
*  CE2_XTS_AES
*
* @param Tweak_ptr [in/out] - Pointer to buffer containing:
*   - Tweak Value (if IsTweakInternal = FALSE) OR 
*   - Sector Number (if IsTweakInternal = TRUE)
*     \note In case tweak value is generated internally it's returned through this parameter
* @param Key_ptr [in] - Pointer to buffer containing: 
*   - AES key used for XTS-AES cipher (if IsTweakInternal = FALSE) OR 
*   - Both: AES Key + Tweak Key concatinated (if TweakInternal = TRUE)
* @param KeySize [in] - Size of Tweak Key and/or AES keys (size of every single key but not size of buffer).
* @param EncryptDecryptFlag [in] - This flag specifies encrypt or decrypt operation to be performed.
* @param DataIn_ptr [in] - A pointer to input data.
* @param DataInSize [in] - Size of input data. It must be at least 16 bytes.
* @param IsTweakInternal [in] - Flag to indicate if tweak value should be generated internally or is put from outside
* @param DataOut_ptr [out] - A pointer to output data.
*
* @returns \b
*  CE2Error_t  
*  - CE2_OK - on success
*  - Otherwise - error code:
*   - CE2_XTS_AES_ILLEGAL_TWEAK_PTR_ERROR
*   - CE2_XTS_AES_ILLEGAL_KEY_PTR_ERROR
*   - CE2_XTS_AES_ILLEGAL_KEY_SIZE_ERROR
*   - CE2_XTS_AES_ILLEGAL_ENCRYPT_MODE_ERROR
*   - CE2_XTS_AES_ILLEGAL_DATAIN_PTR_ERROR
*   - CE2_XTS_AES_TO_SHORT_DATA_SIZE_ERROR
*   - CE2_XTS_AES_ILLEGAL_DATAOUT_PTR_ERROR
*
* \brief \b 
* Description:
*  This function is used to make XTS-AES operations;
*
*  \b 
* Algorithm:
*  -# Verify input parameters;
*  -# Call low level function LLF_XTS_AES().
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_XTS_AES(CE2_AES_XTS_Tweak_t Tweak_ptr,
                                CE2_AES_Key_t           Key_ptr,
                                CE2_AES_KeySize_t       KeySize,
                                CE2_AES_EncryptMode_t   EncryptDecryptFlag, 
                                DxUint8_t               *DataIn_ptr,        
                                DxUint32_t              DataInSize,         
								EcDxBool_t				IsTweakInternal,
								DxUint8_t               *DataOut_ptr
								);

/**
****************************************************************
* Function Name: 
*  CE2_AES_CCM
*
* @param EncrDecrMode [in] - Enumerator variable defining operation mode (0 - encrypt; 1 - decrypt).
* @param CCM_Key [in] - A buffer, containing the AESCCM key passed by user (predefined size 128 bits).
* @param QFieldSize [in] - Byte-size of formatted field for writing significant bytes of the TextSizeQ 
*                          value. Valid values according to our implementation: [2,3,4]. 
* @param N_ptr [in] - A pointer to Nonce - unique value assigned to all data passed into CCM.
*                     Bytes order - big endian form (MSB is the first).
* @param SizeOfN [in] - The size of the user passed Nonce (in bytes). 	Valid values: 
*                       7 <= SizeOfN <= (15-QFieldSize). 
* @param ADataIn_ptr [in] - A pointer to the additional data buffer. The pointer does 
*                           not need to be aligned. On CSI input mode the pointer must be equal to 
*                           value (0xFFFFFFFC | DataInAlignment). 
* @param ADataInSize [in] - The size of the additional data in bytes;  
* @param TextDataIn_ptr [in] - A pointer to the input text data buffer (plain or cipher according to
*                              encrypt-decrypt mode). The pointer does not need to be aligned.
*                              On CSI input mode the pointer must be equal to value (0xFFFFFFFC | DataInAlignment). 
* @param TextDataInSize [in] - The size of the input text data in bytes;  
* @param TextDataOut_ptr [out] - The output text data pointer (cipher or plain text data). 
* @param TextDataOutSize [in] - The size of the output text data buffer. The size of buffer must be enough 
*                               for output the data:
*                                   - on Encrypt mode: TextDataOutSize >= TextDataInSize + SizeOfT.
*                                   - on Decrypt mode: TextDataOutSize >= TextDataInSize - SizeOfT.
* @param SizeOfT [in] - Size of AES-CCM MAC output T in bytes. Valid values: [4,6,8,10,12,14,16].   
*
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*   - CE2_AES_CCM_ILLEGAL_KEY_PTR_ERROR
*   - CE2_AES_CCM_KEY_SIZE_ERROR
*   - CE2_AES_CCM_ILLEGAL_NONCE_PTR_ERROR
*   - CE2_AES_CCM_ILLEGAL_HEADER_PTR_ERROR
*   - CE2_AES_CCM_ENCRYPT_DECRYPT_FLAG_ERROR
*   - CE2_AES_CCM_ILLEGAL_DATA_IN_PTR_ERROR
*   - CE2_AES_CCM_ILLEGAL_DATA_OUT_PTR_ERROR
*   - CE2_AES_CCM_ILLEGAL_TAG_PTR_ERROR
*   - CE2_AES_CCM_ILLEGAL_TAG_SIZE_PTR_ERROR
*   - CE2_AES_CCM_TAG_SIZE_ERROR
*
* \brief \b 
* Description:
*  This function is used to perform the AES_CCM operation in one integrated process.
*  The function preforms CCM algorithm according to NIST 800-38C.
*
*  \b 
* Algorithm:
*  -# Verify input parameters;
*  -# Call low level function LLF_AES_CCM().
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_AES_CCM(CE2_AES_EncryptMode_t  EncrDecrMode,
                                    CE2_AESCCM_Key_t       CCM_Key,
                                    DxUint8_t              QFieldSize,
                                    DxUint8_t             *N_ptr,
                                    DxUint8_t              SizeOfN,
                                    DxUint8_t             *ADataIn_ptr,
                                    DxUint32_t             ADataInSize,
                                    DxUint8_t             *TextDataIn_ptr,
                                    DxUint32_t             TextDataInSize,
                                    DxUint8_t             *TextDataOut_ptr,
                                    DxUint32_t             TextDataOutSize,
                                    DxUint8_t              SizeOfT);

/**
****************************************************************
* Function Name: 
*  CE2_AES_GCM
*
* @param Key_ptr [in] - This is a AES key used for CCM cipher.
* @param KeySize [in] - A size in bytes of the AES key.
* @param IV_ptr [in] - A pointer to buffer with IV (Initial Vector). 
*                      It maybe null if IVSize = 0.
* @param IVSize [in] - A size of IV buffer in bytes;
* @param AAD_ptr [in] - A pointer to AAD (Additional Authentication Data) buffer. 
*                       It maybe null if AADSize = 0.
* @param AADSize [in] - A size of AAD buffer in bytes.
* @param EncryptDecryptFlag [in] - This flag specialize encrypt or decrypt operation. 
* @param DataIn_ptr [in] - A pointer to input data.        
* @param DataInSize [in] - Size of input data.          
* @param DataOut_ptr [out] - A pointer to output data.
* @param Tag_ptr [out] - A pointer to output tag buffer. 
* @param TagSize_ptr [in/out] - A pointer to size of tag buffer.
*                               Note: maximal used size of tag is 16 bytes.
*
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*   - CE2_AES_GCM_ILLEGAL_KEY_PTR_ERROR
*   - CE2_AES_GCM_KEY_SIZE_ERROR
*   - CE2_AES_GCM_ILLEGAL_IV_PTR_ERROR
*   - CE2_AES_GCM_ILLEGAL_AAD_PTR_ERROR
*   - CE2_AES_GCM_ENCRYPT_DECRYPT_FLAG_ERROR
*   - CE2_AES_GCM_ILLEGAL_DATA_IN_PTR_ERROR
*   - CE2_AES_GCM_ILLEGAL_DATA_OUT_PTR_ERROR
*   - CE2_AES_GCM_ILLEGAL_TAG_PTR_ERROR
*   - CE2_AES_GCM_ILLEGAL_TAG_SIZE_PTR_ERROR
*
* \brief \b 
* Description:
*  This function is used to make GCM operations;
*
*  \b 
* Algorithm:
*  -# Verify input parameters;
*  -# Call low level function LLF_AES_GCM().
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_AES_GCM(CE2_AES_Key_t         Key_ptr,
                                 CE2_AES_KeySize_t     KeySize,
                                 DxUint8_t             *IV_ptr, 
                                 DxUint32_t            IVSize,
                                 DxUint8_t             *AAD_ptr, 
                                 DxUint32_t            AADSize,
                                 CE2_AES_EncryptMode_t EncryptDecryptFlag, 
                                 DxUint8_t             *DataIn_ptr,        
                                 DxUint32_t            DataInSize,         
                                 DxUint8_t             *DataOut_ptr,
                                 DxUint8_t             *Tag_ptr, 
                                 DxUint32_t            *TagSize_ptr);

/**
****************************************************************
* Function Name: 
*  CE2_XTS_AES_OLD suport for old version of CryptoRef
*
* @param Key1_ptr [in] - This parameter is a first AES key used for XTS-AES cipher.
* @param Key2_ptr [in] - This parameter is a second AES key used for XTS-AES cipher.
* @param KeySize [in] - Size of first and second AES keys.
* @param SectorNumber [in] - 64-bit sector number.
* @param EncryptDecryptFlag [in] - This flag specialize encrypt or decrypt operation.
* @param DataIn_ptr [in] - A pointer to input data.
* @param DataInSize [in] - Size of input data. It must be at least 16 bytes.
* @param DataOut_ptr [out] - A pointer to output data.
*
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*   - CE2_XTS_AES_ILLEGAL_KEY_PTR_ERROR
*   - CE2_XTS_AES_ILLEGAL_KEY_SIZE_ERROR
*   - CE2_XTS_AES_ILLEGAL_ENCRYPT_MODE_ERROR
*   - CE2_XTS_AES_ILLEGAL_DATAIN_PTR_ERROR
*   - CE2_XTS_AES_TO_SHORT_DATA_SIZE_ERROR
*   - CE2_XTS_AES_ILLEGAL_DATAOUT_PTR_ERROR
*
* \brief \b 
* Description:
*  This function is used to make XTS-AES operations;
*
*  \b 
* Algorithm:
*  -# Verify input parameters;
*  -# Call low level function LLF_XTS_AES().
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_XTS_AES_OLD(CE2_AES_Key_t           Key1_ptr,
									CE2_AES_Key_t           Key2_ptr,
									CE2_AES_KeySize_t       KeySize,
									DxUint64_t				SectorNumber,
									CE2_AES_EncryptMode_t   EncryptDecryptFlag, 
									DxUint8_t               *DataIn_ptr,        
									DxUint32_t              DataInSize,         
									DxUint8_t               *DataOut_ptr);
#ifdef __cplusplus
}
#endif

#endif
